<?php
header('Content-Type: application/json');

include '../functions.php';
include '../../connect.php';
include '../../company-details.php';

// Fetch Token
$tokenData = getEversendToken();
if (!$tokenData) {
    http_response_code(500);
    echo json_encode(['status' => 'error', 'message' => 'Failed to fetch authentication token']);
    exit();
}

// Allow only POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['status' => 'error', 'message' => 'Method not allowed']);
    exit();
}

// Get and decode JSON input
$json = file_get_contents('php://input');
$data = json_decode($json, true);

if (json_last_error() !== JSON_ERROR_NONE) {
    http_response_code(400);
    echo json_encode(['status' => 'error', 'message' => 'Invalid JSON data']);
    exit();
}

// Define required fields
$required_fields = ['sourceWallet', 'amount', 'type', 'destinationCountry', 'destinationCurrency', 'amountType', 'uuid'];

foreach ($required_fields as $field) {
    if (!isset($data[$field]) || empty(trim($data[$field]))) {
        http_response_code(400);
        echo json_encode(['status' => 'error', 'message' => "$field is required"]);
        exit();
    }
}

// Fetch Service fee for exchange
$withdraw_fee = getCompanySetting('withdraw_fee');

// Sanitize and validate inputs
$sourceWallet = filter_var(trim($data['sourceWallet']), FILTER_SANITIZE_STRING);
$type = filter_var(trim($data['type']), FILTER_SANITIZE_STRING);
$uuid = filter_var(trim($data['uuid']), FILTER_SANITIZE_STRING);
$destinationCurrency = filter_var(trim($data['destinationCurrency']), FILTER_SANITIZE_STRING);
$destinationCountry = filter_var(trim($data['destinationCountry']), FILTER_SANITIZE_STRING);
$amountType = filter_var(trim($data['amountType']), FILTER_SANITIZE_STRING);
$amount = filter_var($data['amount'], FILTER_VALIDATE_FLOAT);

// if (!$amount || $amount < 5000) {
//     http_response_code(400);
//     echo json_encode(['status' => 'error', 'message' => 'Minimum withdraw amount is 5000']);
//     exit();
// }

// if ($amount > 2000000) {
//     http_response_code(400);
//     echo json_encode(['status' => 'error', 'message' => 'Maximum withdraw amount is 2000000']);
//     exit();
// }

$service_fee = number_format($amount * $withdraw_fee, 2, '.', '');
$totalAmountRequired = $amount + $service_fee;

// Fetch User's Wallet Balance
$fetchSql = "SELECT amount FROM wallets WHERE uuid = :uuid AND currency = :currency LIMIT 1";
$fetchStmt = $conn->prepare($fetchSql);
$fetchStmt->bindParam(':uuid', $uuid, PDO::PARAM_STR);
$fetchStmt->bindParam(':currency', $sourceWallet, PDO::PARAM_STR);
if (!$fetchStmt->execute()) {
    http_response_code(500);
    echo json_encode(['status' => 'error', 'message' => 'Database query failed']);
    exit();
}
$result = $fetchStmt->fetch(PDO::FETCH_ASSOC);
$wallet_balance = $result ? $result['amount'] : 0;

// Initialize cURL request
$curl = curl_init();
curl_setopt_array($curl, [
    CURLOPT_URL => "https://api.eversend.co/v1/payouts/quotation",
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
    CURLOPT_CUSTOMREQUEST => 'POST',
    CURLOPT_POSTFIELDS => json_encode([
        'sourceWallet' => $sourceWallet,
        'amount' => $amount,
        'type' => $type,
        'destinationCountry' => $destinationCountry,
        'destinationCurrency' => $destinationCurrency,
        'amountType' => $amountType
    ]),
    CURLOPT_HTTPHEADER => [
        'Content-Type: application/json',
        "Authorization: Bearer {$tokenData}"
    ],
]);

// Execute API request
$response = curl_exec($curl);
$http_status = curl_getinfo($curl, CURLINFO_HTTP_CODE);
$error = curl_error($curl);
curl_close($curl);

if ($error) {
    http_response_code(500);
    echo json_encode(['status' => 'error', 'message' => "cURL Error: $error"]);
    exit();
}

// Decode response
$eversendResponse = json_decode($response, true);
if (json_last_error() !== JSON_ERROR_NONE || !isset($eversendResponse['data']['quotation'])) {
    http_response_code(500);
    $eversendError = json_decode($response, true);
    $errorMessage = $eversendError['message'];
    echo json_encode(['status' => 'error', 'message' => $errorMessage]);
    exit();
}

// Extract required values from Eversend response
$eversendFees = $eversendResponse['data']['quotation']['totalFees'];
$total_fees = $eversendFees + $service_fee;
$rate = $eversendResponse['data']['quotation']['exchangeRate'];
$finalTotalAmountRequired = $amount + $total_fees;

// Check wallet balance
if ($wallet_balance < $finalTotalAmountRequired) {
    http_response_code(400);
    echo json_encode([
        'code' => 400,
        'status' => 'error',
        'message' => 'Insufficient wallet balance',
        'rate' => $rate,
        'totalAmountRequired' => $finalTotalAmountRequired,
        'service_fee' => $service_fee,
        'withdraw_fee' => 1000,
        'total_fees' => $total_fees
    ]);
    exit();
}

// Append service fee to response
$eversendResponse['data']['quotation']['service_fee'] = $service_fee;
$eversendResponse['data']['quotation']['total_fees'] = $total_fees;
$eversendResponse['data']['quotation']['totalAmountRequired'] = $finalTotalAmountRequired;
$eversendResponse['data']['quotation']['rate'] = $rate;

// Return updated response
http_response_code($http_status);
echo json_encode($eversendResponse);
?>
